// Constants
const SITE_URL = window.location.hostname === 'localhost' ? 'http://localhost' : 'https://esatcantravel.net';

// Initialize
document.addEventListener('DOMContentLoaded', init);

async function init() {
    // Check if user is logged in
    const user = await chrome.storage.local.get(['user', 'sessionToken']);
    
    if (user.user && user.sessionToken) {
        showMainContent(user.user);
    } else {
        showLoginScreen();
    }
}

// Show Login Screen
function showLoginScreen() {
    document.getElementById('loginScreen').style.display = 'block';
    document.getElementById('mainContent').style.display = 'none';
    
    // Login form
    document.getElementById('loginForm').addEventListener('submit', handleLogin);
    
    // Open site link
    document.getElementById('openSiteLogin').addEventListener('click', (e) => {
        e.preventDefault();
        chrome.tabs.create({ url: `${SITE_URL}/giris.php` });
    });
}

// Handle Login
async function handleLogin(e) {
    e.preventDefault();
    
    const username = document.getElementById('username').value;
    const password = document.getElementById('password').value;
    
    const btn = e.target.querySelector('button');
    btn.textContent = 'Giriş yapılıyor...';
    btn.disabled = true;
    
    try {
        const response = await fetch(`${SITE_URL}/api/extension-login.php`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ username, password })
        });
        
        const data = await response.json();
        
        if (data.success) {
            // Save user data
            await chrome.storage.local.set({
                user: data.user,
                sessionToken: data.token
            });
            
            showMainContent(data.user);
        } else {
            alert(data.message || 'Giriş başarısız!');
            btn.textContent = 'Giriş Yap';
            btn.disabled = false;
        }
    } catch (error) {
        console.error('Login error:', error);
        alert('Bağlantı hatası! Lütfen tekrar deneyin.');
        btn.textContent = 'Giriş Yap';
        btn.disabled = false;
    }
}

// Show Main Content
function showMainContent(user) {
    document.getElementById('loginScreen').style.display = 'none';
    document.getElementById('mainContent').style.display = 'block';
    
    // Set user info
    document.getElementById('userName').textContent = user.full_name || user.username;
    document.getElementById('userType').textContent = user.user_type || 'User';
    
    // Show/hide elements based on user type
    const isUser = user.user_type === 'user';
    const isAdmin = ['admin', 'manager', 'superadmin'].includes(user.user_type);
    
    // Show Kargo & Para Transferi for user accounts only
    if (isUser) {
        document.querySelectorAll('.user-only').forEach(el => {
            el.style.display = '';
        });
    }
    
    // Show Admin Panel button for admin/manager
    if (isAdmin) {
        document.querySelectorAll('.admin-only').forEach(el => {
            el.style.display = '';
        });
    }
    
    // Setup tabs
    setupTabs();
    
    // Setup quick actions
    setupQuickActions();
    
    // Setup home cards
    setupHomeCards();
    
    // Setup forms
    if (isUser) {
        setupCargoForm();
        setupTransferForm();
    }
    
    // Load content
    loadFavorites();
    loadReservations();
    loadCurrency();
    
    // Logout button
    document.getElementById('logoutBtn').addEventListener('click', handleLogout);
}

// Setup Tabs
function setupTabs() {
    const tabs = document.querySelectorAll('.tab');
    const tabContents = document.querySelectorAll('.tab-content');
    
    tabs.forEach(tab => {
        tab.addEventListener('click', () => {
            tabs.forEach(t => t.classList.remove('active'));
            tabContents.forEach(tc => tc.classList.remove('active'));
            
            tab.classList.add('active');
            const targetTab = tab.dataset.tab;
            document.getElementById(`${targetTab}-tab`).classList.add('active');
        });
    });
}

// Setup Quick Actions
function setupQuickActions() {
    // Open Site button
    document.getElementById('openSiteBtn').addEventListener('click', () => {
        chrome.tabs.create({ url: `${SITE_URL}/index.php` });
    });
    
    // Live Support button
    document.getElementById('liveSupport').addEventListener('click', () => {
        chrome.tabs.create({ url: `${SITE_URL}/canli-destek.php` });
    });
    
    // Admin Panel button (only visible for admin/manager)
    const adminBtn = document.getElementById('adminPanel');
    if (adminBtn) {
        adminBtn.addEventListener('click', () => {
            chrome.tabs.create({ url: `${SITE_URL}/admin.php` });
        });
    }
}

// Setup Home Cards
function setupHomeCards() {
    const cards = document.querySelectorAll('.home-card');
    cards.forEach(card => {
        card.addEventListener('click', () => {
            const url = card.dataset.url;
            if (url) {
                chrome.tabs.create({ url: `${SITE_URL}/${url}` });
            }
        });
    });
}

// Setup Cargo Form
function setupCargoForm() {
    const directionSelect = document.getElementById('cargoDirection');
    const step1 = document.getElementById('cargoStep1');
    const step2 = document.getElementById('cargoStep2');
    const step3 = document.getElementById('cargoStep3');
    const flightsList = document.getElementById('flightsList');
    const form = document.getElementById('cargoForm');
    const message = document.getElementById('cargoMessage');
    
    let selectedFlight = null;
    
    // Step 1: Direction selection
    directionSelect.addEventListener('change', async () => {
        const direction = directionSelect.value;
        if (!direction) return;
        
        // Load flights
        flightsList.innerHTML = '<div class="loading">Uçuşlar yükleniyor...</div>';
        step2.style.display = 'block';
        
        try {
            const response = await fetch(`${SITE_URL}/api/get-flights.php?direction=${direction}&status=active`);
            const data = await response.json();
            
            if (data.success && data.flights && data.flights.length > 0) {
                flightsList.innerHTML = data.flights.map(flight => `
                    <div class="flight-item" data-flight-id="${flight.id}" data-flight='${JSON.stringify(flight)}'>
                        <div class="flight-date">📅 ${new Date(flight.flight_date).toLocaleDateString('tr-TR')}</div>
                        <div class="flight-details">
                            <span>${flight.direction === 'istanbul_to_jakarta' ? '🇹🇷 → 🇮🇩' : '🇮🇩 → 🇹🇷'}</span>
                            <span class="flight-capacity">${flight.baggage_limit} kg mevcut</span>
                        </div>
                    </div>
                `).join('');
                
                // Add click handlers
                document.querySelectorAll('.flight-item').forEach(item => {
                    item.addEventListener('click', () => {
                        document.querySelectorAll('.flight-item').forEach(i => i.classList.remove('selected'));
                        item.classList.add('selected');
                        
                        selectedFlight = JSON.parse(item.dataset.flight);
                        document.getElementById('selectedFlightId').value = selectedFlight.id;
                        document.getElementById('selectedDirection').value = direction;
                        
                        step3.style.display = 'block';
                    });
                });
            } else {
                flightsList.innerHTML = '<div class="empty-state"><p>Bu yöne aktif uçuş bulunamadı</p></div>';
            }
        } catch (error) {
            flightsList.innerHTML = '<div class="empty-state"><p>❌ Uçuşlar yüklenemedi</p></div>';
        }
    });
    
    // Step 3: Form submission
    form.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const btn = form.querySelector('button[type="submit"]');
        const originalText = btn.textContent;
        btn.textContent = 'Gönderiliyor...';
        btn.disabled = true;
        
        // Get user data from storage
        const userData = await chrome.storage.local.get('user');
        const user = userData.user;
        
        // Generate request ID
        const requestId = 'ECT' + Date.now() + Math.floor(Math.random() * 1000);
        
        // Calculate price (base price + 12% commission)
        const unitPriceUSD = parseFloat(selectedFlight.price_per_kg) || 0;
        const weight = parseFloat(form.weight.value);
        const basePriceUSD = weight * unitPriceUSD;
        const commissionUSD = basePriceUSD * 0.12;
        const totalPriceUSD = basePriceUSD + commissionUSD;
        
        const formData = {
            requestId: requestId,
            type: 'cargo_request',  // API'de beklenen format
            direction: selectedFlight.direction,
            date: selectedFlight.flight_date,
            flightId: selectedFlight.id,
            email: user.email,
            phone: form.phone.value,
            countryCode: selectedFlight.direction === 'tr-to-id' ? '+90' : '+62',
            weight: weight,
            content: form.content.value,
            totalAmount: totalPriceUSD,
            currency: 'USD',
            timestamp: new Date().toISOString(),
            status: 'pending_payment'
        };
        
        try {
            const response = await fetch(`${SITE_URL}/api/cargo-request.php`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData)
            });
            
            const data = await response.json();
            
            if (data.success) {
                message.textContent = '✅ Kargo isteğiniz alındı! Sipariş No: ' + data.requestId;
                message.className = 'form-message success';
                message.style.display = 'block';
                
                // Reset form
                form.reset();
                step1.style.display = 'block';
                step2.style.display = 'none';
                step3.style.display = 'none';
                directionSelect.value = '';
                selectedFlight = null;
            } else {
                throw new Error(data.message || 'Bir hata oluştu');
            }
        } catch (error) {
            message.textContent = '❌ ' + error.message;
            message.className = 'form-message error';
            message.style.display = 'block';
        } finally {
            btn.textContent = originalText;
            btn.disabled = false;
            setTimeout(() => {
                message.style.display = 'none';
            }, 5000);
        }
    });
}

// Setup Transfer Form
function setupTransferForm() {
    const form = document.getElementById('transferForm');
    const message = document.getElementById('transferMessage');
    
    form.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const btn = form.querySelector('button[type="submit"]');
        const originalText = btn.textContent;
        btn.textContent = 'Gönderiliyor...';
        btn.disabled = true;
        
        const formData = {
            name: form.name.value,
            email: form.email.value,
            phone: form.phone.value,
            direction: form.direction.value,
            amount: form.amount.value,
            notes: form.notes.value,
            type: 'money_transfer'
        };
        
        try {
            const response = await fetch(`${SITE_URL}/api/money-transfer-request.php`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData)
            });
            
            const data = await response.json();
            
            if (data.success) {
                message.textContent = '✅ Transfer talebiniz alındı! En kısa sürede dönüş yapılacak.';
                message.className = 'form-message success';
                message.style.display = 'block';
                form.reset();
            } else {
                throw new Error(data.message || 'Bir hata oluştu');
            }
        } catch (error) {
            message.textContent = '❌ ' + error.message;
            message.className = 'form-message error';
            message.style.display = 'block';
        } finally {
            btn.textContent = originalText;
            btn.disabled = false;
            setTimeout(() => {
                message.style.display = 'none';
            }, 5000);
        }
    });
}

// Load Favorites
async function loadFavorites() {
    const result = await chrome.storage.local.get('favorites');
    const favorites = result.favorites || [];
    
    const container = document.getElementById('favoritesList');
    
    if (favorites.length === 0) {
        container.innerHTML = '<div class="empty-state"><span class="empty-icon">⭐</span><p>Henüz favori tur eklemediniz</p></div>';
        return;
    }
    
    container.innerHTML = favorites.map(tour => `
        <div class="list-item" onclick="window.open('${SITE_URL}/tour-detail.php?id=${tour.id}')">
            <div class="item-title">${tour.title}</div>
            <div class="item-meta">
                <span>📍 ${tour.island || 'Endonezya'}</span>
                <span class="item-price">${tour.price || 'Fiyat Sor'}</span>
            </div>
        </div>
    `).join('');
}

// Load Reservations
async function loadReservations() {
    const container = document.getElementById('reservationsList');
    
    try {
        const userData = await chrome.storage.local.get('user');
        const response = await fetch(`${SITE_URL}/api/get-user-reservations.php?user_id=${userData.user.id}`);
        const data = await response.json();
        
        if (data.success && data.reservations.length > 0) {
            container.innerHTML = data.reservations.map(res => `
                <div class="list-item">
                    <div class="item-title">${res.name}</div>
                    <div class="item-meta">
                        <span>${new Date(res.created_at).toLocaleDateString('tr-TR')}</span>
                        <span class="item-status ${res.status}">${getStatusText(res.status)}</span>
                    </div>
                    <div class="item-meta">
                        <span>📧 ${res.email}</span>
                        <span>📞 ${res.phone}</span>
                    </div>
                </div>
            `).join('');
        } else {
            container.innerHTML = '<div class="empty-state"><span class="empty-icon">📋</span><p>Henüz rezervasyonunuz yok</p></div>';
        }
    } catch (error) {
        console.error('Reservations error:', error);
        container.innerHTML = '<div class="empty-state"><span class="empty-icon">❌</span><p>Rezervasyonlar yüklenemedi</p></div>';
    }
}

// Load Currency
async function loadCurrency() {
    try {
        const response = await fetch(`${SITE_URL}/api/get-exchange-rates.php`);
        const data = await response.json();
        
        if (data.success && data.rates) {
            // 1000 IDR to TRY
            const idrToTry = (data.rates.IDR_TRY || (data.rates.USD_TRY / data.rates.USD_IDR)) * 1000;
            document.getElementById('idrToTry').textContent = idrToTry.toFixed(2) + ' ₺';
            
            document.getElementById('rateUpdated').textContent = new Date(data.updated_at).toLocaleString('tr-TR', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
            
            // 1 USD to TRY
            const usdToTry = data.rates.USD_TRY;
            document.getElementById('usdToTry').textContent = usdToTry.toFixed(2) + ' ₺';
            
            // 1 USD to IDR
            const usdToIdr = data.rates.USD_IDR;
            document.getElementById('usdToIdr').textContent = usdToIdr.toLocaleString('tr-TR', {
                minimumFractionDigits: 0,
                maximumFractionDigits: 0
            }) + ' Rp';
            
            // Save to storage
            await chrome.storage.local.set({ 
                exchangeRates: data.rates,
                ratesUpdated: Date.now() 
            });
        } else {
            throw new Error('Kurlar alınamadı');
        }
    } catch (error) {
        console.error('Currency error:', error);
        document.getElementById('idrToTry').textContent = 'N/A';
        document.getElementById('usdToTry').textContent = 'N/A';
        document.getElementById('usdToIdr').textContent = 'N/A';
        document.getElementById('rateUpdated').textContent = 'Güncellenemedi';
    }
    
    // Refresh button
    document.getElementById('refreshRate').addEventListener('click', () => {
        document.getElementById('refreshRate').textContent = '🔄 Yenileniyor...';
        loadCurrency().then(() => {
            document.getElementById('refreshRate').textContent = '🔄 Kurları Yenile';
        });
    });
}

// Handle Logout
async function handleLogout() {
    if (confirm('Çıkış yapmak istediğinize emin misiniz?')) {
        await chrome.storage.local.remove(['user', 'sessionToken']);
        window.location.reload();
    }
}

// Helper: Get Status Text
function getStatusText(status) {
    const statuses = {
        'pending': 'Beklemede',
        'confirmed': 'Onaylandı',
        'cancelled': 'İptal',
        'completed': 'Tamamlandı'
    };
    return statuses[status] || status;
}

// Listen for messages (e.g., favorite added from content script)
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.action === 'favoriteAdded' || message.action === 'favoriteRemoved') {
        loadFavorites();
    }
});
